	package gov.va.genisis2.bs.service.impl;

import java.util.List;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.stereotype.Service;

import gov.va.genisis2.bs.converter.BookmarkConverter;
import gov.va.genisis2.bs.converter.LabelConverter;
import gov.va.genisis2.bs.data.model.Bookmark;
import gov.va.genisis2.bs.data.model.Label;
import gov.va.genisis2.bs.data.repository.BookmarkRepository;
import gov.va.genisis2.bs.data.repository.LabelRepository;
import gov.va.genisis2.bs.service.IBusinessService;
import gov.va.genisis2.ts.common.dto.BookmarkDTO;
import gov.va.genisis2.ts.common.dto.LabelDTO;
import gov.va.genisis2.ts.common.enums.ErrorEnum;
import gov.va.genisis2.ts.common.exception.TSDuplicateDataException;
import gov.va.genisis2.ts.common.exception.TSNoDataFoundException;
import gov.va.genisis2.ts.common.exception.TSRuntimeException;

/**
 * Implementation for Bookmark Service calls
 * @author PII
 */
@Service
public class BusinessService implements IBusinessService {
	
	private static final Logger LOGGER = LogManager.getLogger(BusinessService.class);
	
	@Autowired
	private BookmarkRepository bookmarkRepository;
	
	@Autowired
	private LabelRepository labelRepository;
	
	@Autowired
	private BookmarkConverter bookmarkConverter;
	
	@Autowired
	private LabelConverter labelConverter;
	
	@Override
	@Transactional
	public List<BookmarkDTO> getBookmarks() {
		List<Bookmark> list = bookmarkRepository.findAll();
		
		if( list == null )
			throw new TSRuntimeException("Error getting bookmark list");
		
		return list.stream()
				.map(bookmark -> bookmarkConverter.convert(bookmark))
				.collect(Collectors.toList());
	}
	
	@Override
	@Transactional
	public BookmarkDTO createBookmark(BookmarkDTO bookmarkDto) {
		Bookmark savedBookmark = null;
		
		//Convert the DTO to a model object
		Bookmark bookmark = bookmarkConverter.convert(bookmarkDto);
		
		try {
			//Save the bookmark model
			savedBookmark = bookmarkRepository.save(bookmark);
		} catch (DataIntegrityViolationException ex) {
			LOGGER.error(String.format(ErrorEnum.BOOKMARK_DUPLICATE_ERROR.getErrorMessage(), 
					bookmarkDto.getConceptUri(), bookmarkDto.getUsername()), ex);
			throw new TSDuplicateDataException(
					String.format(ErrorEnum.BOOKMARK_DUPLICATE_ERROR.getErrorMessage(), 
							bookmarkDto.getConceptUri(), bookmarkDto.getUsername()), ex);
		} catch (Exception e) {
			LOGGER.error(ErrorEnum.INTERNAL_SYS_ERROR, e);
			throw new TSRuntimeException(ErrorEnum.INTERNAL_SYS_ERROR.toString());
		}
		return bookmarkConverter.convert(savedBookmark);
	}
	
	@Override
	@Transactional
	public void deleteBookmark(Integer id) {
		Bookmark bookmark = null;
		try {
			bookmark = bookmarkRepository.findOne(id);
			if (bookmark != null) {
				if (bookmark.getLabels() != null) {
					// Delete associated labels
					for (Label l : bookmark.getLabels()) {
						labelRepository.delete(l.getLabelId());
					}
				}
				// Finally delete the bookmark
				bookmarkRepository.delete(bookmark.getBookmarkId());
			}
			else {
				//bookmark does not exist
				throw new TSNoDataFoundException(
						String.format(ErrorEnum.BOOKMARK_DOESNT_EXIST.getErrorMessage(), id ));
			}
		} catch (IllegalArgumentException e) {
			LOGGER.error("Id: " + id + " passed is null");
			throw new TSRuntimeException(ErrorEnum.INTERNAL_SYS_ERROR.toString(), e);
		}
	}
	
	@Override
	public BookmarkDTO labelBookmark(LabelDTO labelDto) {
		
		//check if the bookmark exists
		Bookmark bookmark = bookmarkRepository.findOne(labelDto.getBookmarkId());
		if( bookmark == null )
			throw new TSNoDataFoundException(String.format(ErrorEnum.BOOKMARK_DOESNT_EXIST.getErrorMessage(),
					labelDto.getBookmarkId()));
		
		Label label = labelConverter.convert(labelDto);
		
		label.setBookmark(bookmark);
		
		//instead of just returning the label..return the bookmark and associated labels
		Label savedLabel = labelRepository.save(label) ;
		Bookmark updatedBookmark = bookmarkRepository.findOne(savedLabel.getBookmarkId());
		
		if( updatedBookmark == null ) //really shouldn't happen
			throw new TSNoDataFoundException(String.format(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage(),
					savedLabel.getBookmarkId()));
		
		return bookmarkConverter.convert(updatedBookmark);
	}
	
	@Override
	public BookmarkDTO unlabelBookmark(LabelDTO labelDto) {

		Bookmark bookmark = bookmarkRepository.findOne(labelDto.getBookmarkId());
		
		//check if the bookmark exists
		if( bookmark == null )
			throw new TSNoDataFoundException(String.format(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage(), 
					labelDto.getBookmarkId()));
		
		Label label = bookmark.getLabels()
		.stream()
		.filter( l -> l.getName().equalsIgnoreCase(labelDto.getName()))
		.findFirst().orElse(null);
		
		if( label == null )
			throw new TSNoDataFoundException(String.format(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage(),
					labelDto.getName())); 
		
		labelRepository.delete(label.getLabelId());
		
		//instead of returning the deleted label return the updated bookmark and should NOT 
		//show the deleted label
		Bookmark updatedBookmark = bookmarkRepository.findOne(bookmark.getBookmarkId());
		
		if( updatedBookmark == null )
			throw new TSNoDataFoundException(String.format(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage(),
					bookmark.getBookmarkId()));
		
		return bookmarkConverter.convert(updatedBookmark);
	}
	
	@Override
	public List<BookmarkDTO> fetchAllBookmarksByUsername(String username) {
		
		List<Bookmark> bookmarks = bookmarkRepository.getAllBookmarksAndLabelsGivenUsername(username);
		
		if( bookmarks == null )
			throw new TSNoDataFoundException(String.format(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage(), username));
		
		return bookmarks.stream()
		.map(bookmark -> bookmarkConverter.convert(bookmark))
		.collect(Collectors.toList());
	}
	
	@Override
	public List<BookmarkDTO> fetchAllBookmarksByLabel(String label) {
		List<Bookmark> bookmarks = bookmarkRepository.getAllBookmakrsGivenLabel(label);
		
		if( bookmarks == null )
			throw new TSNoDataFoundException(String.format(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage(), label));
		
		return bookmarks.stream()
				.map(bookmark -> bookmarkConverter.convert(bookmark))
				.collect(Collectors.toList());
	}
	
	@Override
	public List<LabelDTO> fetchAllLabelsByUsername(String username){
	
		List<Label> labels = labelRepository.getAllLabelsGivenUsername(username);
		
		if( labels == null )
			throw new TSNoDataFoundException(String.format(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage(), username));
		
		return labels.stream()
				.map(label -> labelConverter.convert(label))
				.collect(Collectors.toList());
	}
//	@Override
//	public BookmarkDTO getBookmark(Integer id) {
//		Bookmark bookmark = bookmarkRepository.findOne(id);
//		
//		if( bookmark == null ) {
//			throw new TSNoDataFoundException(String.format(ErrorEnum.NO_DATA_FOUND_ERROR.getErrorMessage(), id));
//		}
//		
//		return bookmarkConverter.convert(bookmark);
//	}
	public void setBookmarkConverter( BookmarkConverter bookmarkConverter) {
		this.bookmarkConverter = bookmarkConverter;
	}
	
	public void setLabelConverter( LabelConverter labelConverter) {
		this.labelConverter = labelConverter;
	}

}
